let connected = false;
let account = null;

// 配置允许连接的域名
const ALLOWED_DOMAIN = 'chainguardlabs.com';

// 指定的连接网站
const CONNECT_URL = 'https://chainguardlabs.com';

document.addEventListener('DOMContentLoaded', function() {
  const connectBtn = document.getElementById('connect-btn');
  const walletStatus = document.getElementById('wallet-status');
  const protectionStatus = document.getElementById('protection-status');
  
  // 初始检查
  checkInitialState();
  
  connectBtn.addEventListener('click', async function() {
    if (!connected) {
      await handleConnect();
    } else {
      disconnectWallet();
    }
  });
  
  async function checkInitialState() {
    // 从存储中获取连接状态
    chrome.storage.local.get(['connected', 'account'], (result) => {
      if (result.connected && result.account) {
        updateConnectedState(result.account);
      }
    });
  }
  
  async function handleConnect() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      
      // 检查当前页面是否可以注入脚本
      if (!tab.url || tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://')) {
        showDomainGuideMessage();
        return;
      }
      
      // 检查是否在允许的域名下
      const currentUrl = new URL(tab.url);
      const currentDomain = currentUrl.hostname;
      
      if (!isAllowedDomain(currentDomain)) {
        showDomainGuideMessage();
        return;
      }
      
      connectBtn.textContent = 'Connecting...';
      connectBtn.disabled = true;
      connectBtn.classList.add('loading');
      
      // 设置超时计时器
      const timeoutId = setTimeout(() => {
        showError('Connection timeout. Please ensure MetaMask is installed.');
        connectBtn.textContent = 'Connect Wallet';
        connectBtn.disabled = false;
        connectBtn.classList.remove('loading');
      }, 5000);
      
      try {
        // 先检查是否有钱包
        const checkResults = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: checkMetaMask,
          world: 'MAIN'
        });
        
        const hasWallet = checkResults[0].result.hasWallet;
        
        if (!hasWallet) {
          clearTimeout(timeoutId);
          showError('MetaMask not detected. Please install it first.');
          connectBtn.textContent = 'Connect Wallet';
          connectBtn.disabled = false;
          connectBtn.classList.remove('loading');
          return;
        }
        
        // 尝试连接钱包
        const results = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: connectMetaMask,
          world: 'MAIN'
        });
        
        clearTimeout(timeoutId);
        
        const result = results[0].result;
        
        if (result.error) {
          showError('Connection failed: ' + result.error);
        } else if (result.account) {
          // 保存连接状态和域名
          chrome.storage.local.set({ 
            connected: true, 
            account: result.account,
            connectedDomain: currentDomain
          });
          updateConnectedState(result.account, currentDomain);
        }
      } catch (error) {
        clearTimeout(timeoutId);
        console.error('Script execution error:', error);
        showError('Please use this feature on a regular webpage.');
      }
      
      connectBtn.textContent = 'Connect Wallet';
      connectBtn.disabled = false;
      connectBtn.classList.remove('loading');
      
    } catch (error) {
      console.error('Connection error:', error);
      showError('Connection failed. Please try again.');
      connectBtn.textContent = 'Connect Wallet';
      connectBtn.disabled = false;
      connectBtn.classList.remove('loading');
    }
  }
  
  // 检查是否是允许的域名
  function isAllowedDomain(domain) {
    return domain === ALLOWED_DOMAIN || domain.endsWith('.' + ALLOWED_DOMAIN);
  }
  
  // 检查MetaMask是否存在
  function checkMetaMask() {
    return {
      hasWallet: typeof window.ethereum !== 'undefined'
    };
  }
  
  // 在页面中执行的连接函数
  function connectMetaMask() {
    return new Promise((resolve) => {
      if (typeof window.ethereum === 'undefined') {
        resolve({ error: 'NO_WALLET' });
        return;
      }
      
      window.ethereum.request({ method: 'eth_requestAccounts' })
        .then(accounts => {
          resolve({ account: accounts[0] });
        })
        .catch(error => {
          if (error.code === 4001) {
            resolve({ error: 'User rejected the connection request' });
          } else {
            resolve({ error: error.message });
          }
        });
    });
  }
  
  // 显示域名限制的引导消息
  function showDomainGuideMessage() {
    const guideHtml = `
      <div class="domain-guide">
        <div class="status-icon">🔒</div>
        <h3>Connect on ChainGuard Labs</h3>
        <p>For security reasons, wallet connection is only available on our official website.</p>
        <div class="allowed-sites">
          <p class="sites-title">Official Website:</p>
          <div class="allowed-site-url">${ALLOWED_DOMAIN}</div>
        </div>
        <button id="go-to-site" class="primary-btn">Go to ChainGuard Labs</button>
      </div>
    `;
    
    walletStatus.innerHTML = guideHtml;
    walletStatus.classList.add('guide-mode');
    
    document.getElementById('go-to-site').addEventListener('click', () => {
      chrome.tabs.create({ url: CONNECT_URL });
    });
  }
  
  function updateConnectedState(address, domain) {
    connected = true;
    account = address;
    
    walletStatus.classList.add('connected');
    walletStatus.classList.remove('guide-mode');
    walletStatus.innerHTML = `
      <div class="status-icon">✅</div>
      <p class="status-text">Wallet Connected</p>
      <p class="status-detail">${address.substring(0, 6)}...${address.substring(38)}</p>
      ${domain ? `<p class="connected-domain">Connected via: ${domain}</p>` : ''}
    `;
    
    connectBtn.textContent = 'Disconnect';
    connectBtn.disabled = false;
    
    protectionStatus.style.display = 'block';
  }
  
  function disconnectWallet() {
    connected = false;
    account = null;
    
    // 清除存储的连接状态
    chrome.storage.local.remove(['connected', 'account', 'connectedDomain']);
    
    walletStatus.classList.remove('connected', 'guide-mode');
    walletStatus.innerHTML = `
      <div class="status-icon">🔗</div>
      <p class="status-text">Not Connected</p>
      <p class="status-detail">Please install MetaMask or similar wallet</p>
    `;
    
    connectBtn.textContent = 'Connect Wallet';
    
    protectionStatus.style.display = 'none';
  }
  
  function showError(message) {
    // 如果是引导消息，先恢复原始状态
    if (walletStatus.querySelector('.domain-guide')) {
      walletStatus.classList.remove('guide-mode');
      walletStatus.innerHTML = `
        <div class="status-icon">🔗</div>
        <p class="status-text">Not Connected</p>
        <p class="status-detail">Please install MetaMask or similar wallet</p>
      `;
    }
    
    const statusDetail = walletStatus.querySelector('.status-detail');
    if (statusDetail) {
      const originalText = statusDetail.textContent;
      statusDetail.textContent = message;
      statusDetail.style.color = '#ff4757';
      
      setTimeout(() => {
        statusDetail.textContent = originalText;
        statusDetail.style.color = '';
      }, 5000);
    }
  }
  
  // 监听存储变化
  chrome.storage.onChanged.addListener(function(changes, namespace) {
    if (changes.connected && changes.connected.newValue) {
      chrome.storage.local.get(['account', 'connectedDomain'], (result) => {
        if (result.account) {
          updateConnectedState(result.account, result.connectedDomain);
        }
      });
    }
  });
});